#include "globals.h"
#include "delay.h"
#include "main.h"
#include "lcd.h"

#include "keypad.h"

static bit	COL4_SELECT @ PORTBIT(PORTD, 3);
static bit	COL3_SELECT @ PORTBIT(PORTD, 2);
static bit	COL2_SELECT @ PORTBIT(PORTD, 1);
static bit	COL1_SELECT @ PORTBIT(PORTD, 0);

static bit	ROW4		@ PORTBIT(PORTD, 7);
static bit	ROW3		@ PORTBIT(PORTD, 6);
static bit	ROW2		@ PORTBIT(PORTD, 5);
static bit	ROW1		@ PORTBIT(PORTD, 4);

u8	LastKey, Keypad, DebouncedKeyASCII;

// scans keypad and returns 'ascii' value in Keypad
// returns NOKEY if none pressed
void	ScanKeypad()
{
	// now do keypad
	LastKey = Keypad;
	Keypad = NOKEY;

	// check COL1
	COL1_SELECT = 1;
	COL2_SELECT = 0;
	COL3_SELECT = 0;
	COL4_SELECT = 0;

	DelayUs(50);
	if (ROW1) Keypad = KEYR1C1;
	if (ROW2) Keypad = KEYR1C2;
	if (ROW3) Keypad = KEYR1C3;
	if (ROW4) Keypad = KEYR1C4;
	
	COL1_SELECT = 0;
	COL2_SELECT = 1;

	DelayUs(50);
	if (ROW1) Keypad = KEYR2C1;
	if (ROW2) Keypad = KEYR2C2;
	if (ROW3) Keypad = KEYR2C3;
	if (ROW4) Keypad = KEYR2C4;
	
	COL2_SELECT = 0;
	COL3_SELECT = 1;
	
	DelayUs(50);
	if (ROW1) Keypad = KEYR3C1;
	if (ROW2) Keypad = KEYR3C2;
	if (ROW3) Keypad = KEYR3C3;
	if (ROW4) Keypad = KEYR3C4;

	COL3_SELECT = 0;
	COL4_SELECT = 1;
	
	DelayUs(50);
	if (ROW1) Keypad = KEYR4C1;
	if (ROW2) Keypad = KEYR4C2;
	if (ROW3) Keypad = KEYR4C3;
	if (ROW4) Keypad = KEYR4C4;

	// check sense line
	if (SenseIn)
	{
		if (Keypad == NOKEY)
			Keypad = KEYNEXT;
	}

	// if change in keypad then update
	if (Keypad != LastKey)
	{
		DebouncedKeyASCII = Keypad;
		DelayMs(50);
		if (Keypad != NOKEY)
		{
			Beep(125, 50);
		}
	}
	else
		DebouncedKeyASCII = NOKEY;
}


// enter a number
// line = display line, pos = display column
// maxdigits is max digits that can be entered
// initial is initial value of number
// dp is position of decimal point (number is returned as fixed point)

int	gEditCancelled;		// true if user cancelled edit, false if number valid

u16		EditNumber(u8 line, u8 pos, u8 maxdigits, u16 initial, u8 dp)
{
	char	InputBuff[5];
	u16 	result = initial;
	u8		CurBuf;
	s8		i;
	u16		dig;
	int		editing = 1;
	int		rdp;
	
	gEditCancelled = false;

	// place number into buffer

	for (i = maxdigits - 1; i >= 0; i--)
	{
		dig = initial / 10;
		dig = initial - (dig * 10);
		initial /= 10;
		InputBuff[i] = (char) dig + '0';
	}

	CurBuf = 0;
	
	// bound number by []'s
	LCDGoto(line, pos - 1);
	LCDPutch('[');
	rdp = dp < maxdigits ? 1 : 0;
	LCDGoto(line, pos + maxdigits + rdp);
	LCDPutch(']');

	// draw contents of input buffer (allowing for dp)
	LCDGoto(line, pos);
	for (i = 0; i < maxdigits; i++)
	{
		if (i == dp)
			LCDPutch('.');
		LCDPutch(InputBuff[i]);
	}
	LCDGoto(line, pos);

	LCDCurserFlash;
	LCDCurserOn;
	
	while(editing)
	{
		ScanKeypad();

		switch (DebouncedKeyASCII)
		{
			case	'0':
			case	'1':
			case	'2':
			case	'3':
			case	'4':
			case	'5':
			case	'6':
			case	'7':
			case	'8':
			case	'9':
				InputBuff[CurBuf] = DebouncedKeyASCII;
				LCDPutch(InputBuff[CurBuf]);
				// fall through to 'next'

			// Next
			case	'N':
				if (CurBuf < (maxdigits - 1))
					CurBuf++;
				rdp = CurBuf >= dp ? 1 : 0;
				LCDGoto(line, pos + CurBuf + rdp);
				break;

			// Previous
			case	'P':
				if (CurBuf > 0)
					CurBuf--;
				rdp = CurBuf >= dp ? 1 : 0;
				LCDGoto(line, pos + CurBuf + rdp);
				break;

			// Cancel
			case	'C':
				editing = 0;
				gEditCancelled = true;
				break;

			// OK
			case	'O':
				CurBuf = 0;
				result = 0;
				while(CurBuf < maxdigits)
				{
					result *= 10;
					result += InputBuff[CurBuf] - '0';
					CurBuf++;
				}
				editing = 0;
				break;

		}
	};

	LCDCurserOff;

	LCDGoto(line, pos - 1);
	LCDPutch(' ');
	rdp = dp < maxdigits ? 1 : 0;
	LCDGoto(line, pos + maxdigits + rdp);
	LCDPutch(' ');

	return result;
}
