/*
 *	LCD interface example
 *	Uses routines from delay.c
 *	This code will interface to a standard LCD controller
 *	like the Hitachi HD44780. It uses it in 4 bit mode, with
 *	the hardware connected as follows (the standard 14 pin 
 *	LCD connector is used):
 *	
 *  To use:	call lcd_init(), then other routines as required.
 *	
 */

#include "globals.h"
#include "lcd.h"
#include "delay.h"

//#define	LCD_STROBE	((LCD_EN = 1),(LCD_EN = 0))
#define	LCD_STROBE	{LCD_EN = 1; DelayUs(2); LCD_EN = 0;}

u8	LCDLine;

/* write a byte to the LCD in 4 bit mode */

void	LCDWriteByte(unsigned char c)
{
	LCDPORT = (LCDPORT & 0x0F) |  (c & 0xf0);
	LCD_STROBE;
	LCDPORT = (LCDPORT & 0x0F) |  (c << 4);
	LCD_STROBE;
	DelayUs(40);
}

/*
 * 	Clear and home the LCD
 */

void	LCDClear(void)
{
	LCD_RS = 0;
	LCDWriteByte(0x1);
	DelayMs(2);
	LCDLine = 0;
}

// clear a section of line
void	LCDClearLine(u8 line, u8 col, u8 num)
{
	int	i;
	LCDGoto(line, col);
	for (i = 0; i < num; i++)
		LCDPutch(' ');
}


/* write a string of chars to the LCD */

void	LCDPuts(const char * s)
{
	u8	col = 0;
	
	LCD_RS = 1;	// write characters
	while(*s)
	{
		// handle CR
		if ((*s) == 10)
		{
			while(col++ < LCDWIDTH)	// clear to end of line
				LCDWriteByte(' ');
				
			LCDGoto(LCDLine + 1, 0);
			s++;
			col = 0;
		}
		else
			LCDWriteByte(*s++);
			
		col++;
	}
}

/* write one character to the LCD */

void	LCDPutch(char c)
{
	LCD_RS = 1;	// write characters
	LCDPORT = (LCDPORT & 0x0F) |  (c & 0xf0);
	LCD_STROBE;
	LCDPORT = (LCDPORT & 0x0F) |  (c << 4);
	LCD_STROBE;
	DelayUs(40);
}


/*
 * Go to the specified position
 */
void	LCDGoto(unsigned char line, unsigned char pos)
{
	unsigned char offset = 0;
	if (line & 1)
		offset += 0x40;
	// on 4x20 line displays, line 0 continues on line 2 etc
	if (line & 2)
		offset += 20;
		
	LCD_RS = 0;
	LCDWriteByte(0x80 + offset + pos);
	LCDLine = line;
}

/* initialise the LCD - put into 4 bit mode */

void	LCDInit(void)
{
	LCD_RS = 0;			// write control bytes
	DelayMs(15);		// power on delay
	LCDPORT = 0x30;		// attention!
	LCD_STROBE;
	DelayMs(5);
	LCD_STROBE;
	DelayUs(100);
	LCD_STROBE;
	DelayMs(5);
	LCDPORT = 0x20;		// set 4 bit mode
	LCD_STROBE;
	DelayUs(40);
	LCDWriteByte(0x28);		// 4 bit mode, 1/16 duty, 5x8 font
	LCDWriteByte(0x08);		// display off
	LCDWriteByte(0x0C);		// display curser off
	LCDWriteByte(0x06);		// entry mode
}

//-------------------------------------------------
// large number stuff...

#ifdef	LARGENUMBERS

const u8 LargeNumLine1[] =
{
	 0,  2,  1,   32,  5, 32,   32,  4,  1,   32,  4,  1,    0,  3,  1,
	 0,  4, 32,    0,  3, 32,   32,  2,  1,    0,  4,  1,    0,  4,  1,
	32,  3, 32,	// minus
	32, 32, 32	// space
};

const u8 LargeNumLine2[] =
{
	 0,  3,  1,   32,  5, 32,    0,  3, 32,   32,  3,  1,   32, 32,  1,
	32,  3,  1,    0,  3,  1,   32,  32, 1,    0,  3,  1,   32, 32,  1,
	32, 32, 32,	
	32, 32, 32	
};

// large number user defined chars
const u8 UDG[] =
{
	0x01, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x01,
	0x10, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x10,
	0x1f, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x1f,
	0x1f, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x1f,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03
};

void	InitLargeNumbers()
{
	u8	i;
	
	LCD_RS = 0;
	LCDWriteByte(0x40);		// char ram addr = 0
	
	for (i = 0; i < 6*8; i++)
		LCDPutch(UDG[i]);
}

void	PrintLargeNumber(u8 num, u8 col)
{
	u8	indx = num * 3;
	
	LCDGoto(0, col);
	LCDPutch(LargeNumLine1[indx]);
	LCDPutch(LargeNumLine1[indx+1]);
	LCDPutch(LargeNumLine1[indx+2]);
	LCDGoto(1, col);
	LCDPutch(LargeNumLine2[indx]);
	LCDPutch(LargeNumLine2[indx+1]);
	LCDPutch(LargeNumLine2[indx+2]);
}

#endif
