#include "globals.h"
#include "display.h"
#include "lcd.h"
#include "keypad.h"
#include "delay.h"
#include "table.h"
//#include "stepratetable.h"

#include "main.h"


//----------------------
// globals 

int	gComplete;

//----------------------

void	OnInit(void);	// initialise all vars and hardware
void	OnTimer(void);	// process timer
void	OnIdle(void);	// process idle

//----------------------

//	Rate = clock / (4 * prescale * (65536 - N))
//  N = 65536 - (clk / (4 * prescale * Rate))
//#define	TIMER_RATE		53036		// 10hz @ 4 mhz clock
#define TIMER_RATE		3036			// 20hz @ 40Mhz clock
#define	SET_TIMER(x)	{TMR1H =  x / 256; TMR1L = x & 255;}
	
#define	DIRECTOUT PORTC	// top 4 bits

//----------------------
// global vars

// current mode
u8		Mode = GREETING;
u16		BeepLength;
int		gProcessTimer = false;
int		gTimerCount = 0;	// 50mS counter

// stepper vars
u16	gStepRate;
u16 gStepRateMax;
u16	gStepRateRamp;
u32	volatile gSteps;	// total steps to take
u16	gAccelSteps;		// number of steps we'll accelerate for
u16	gDecelSteps;		// number of steps we'll decelerate for

//-----------------------
// start here...

main()
{
	// init some global vars
	gComplete = true;
	gSteps = 0;
	
	// init hardware
	OnInit();
	
	// set port state
	AckOut = 0;
	ActiveOut = 1;

	// do forever
	while(1)
	{
		// process any timer events that are queued
		if (gProcessTimer)
		{
			// process any timer events
			OnTimer();
			gProcessTimer = false;
		}
		
		// process menus etc.
		OnIdle();
	}
}

//----------------------------------------
// process idle
void	OnIdle()
{
	ScanKeypad();
	
	switch (Mode)
	{
		case	GREETING:
			LCDClear();
			//		  12345678901234567890
			LCDPuts("Rotary Table Divider");
			LCDGoto(1, 8);
			LCDPuts("V1.1");
			LCDGoto(3, 0);
			LCDPuts("(C) Steve Ward 2006");
			gTimerCount = 0;
			Mode = WAITGREET;
			break;
			
		case	WAITGREET:
			if (gTimerCount >= 30)
				Mode = BEGINTABLE;
			break;

		case	BEGINTABLE:
			InitTable();
			BeginTableMenus();
			break;
			
		case	PROCESSTABLE:
			ProcessTableMenus();
			break;
	}
}

//----------------------------------------
// process timer1 - called twice per second

void	OnTimer()
{
	// handle any timer related 'events' here
}

//----------------------------------------
// initialisation occurs here

void	OnInit()
{
	// IO ports setup
	TRISA = 0xFF;			// input
	PORTA = 0x00;
	
	TRISB = 0x00;			// output for LCD
	PORTB = 0x00;
	
	TRISC = 0x00;
	PORTC = 0x00;
	
	ADCON1 = 7;				// digital inputs on port A

	// port D - 0-3 keypad out, 4-7 keypad in
	TRISD = 0xF0;
	PORTD = 0x00;

	// various bits
	DirectionTris = 0;		// direction line is output
	StepTris = 0;			// as is step line
	ActiveTris = 0;			// and active line
	SenseTris = 1;			// sense is input
	AckTris = 0;			// ack output
	Limit1Tris = 1;			// limit switches inputs
	Limit2Tris = 1;

	// pwm setup (uses timer 2)
	CCP1CON = 0xc;			// set PWM mode and clear low bits of CCPR1
	CCPR1L = 0x7f;			// initial value is zero
	PR2 = 0xff;				// 39.06khz @ 10mhz clock
	T2CON = 0x04;			// enable timer 2 (and set prescale to 1)
	TRISC &= ~0x02;			// pwm pin is output
	
	CKP = 1; CKE = 0;		// clock mode
	SMP = 0;

	SSPEN = 0;				// SPI mode off

	// setup timer1
	SET_TIMER(TIMER_RATE);
	TMR1ON = 1;				// enable timer 1
	TMR1CS = 0;				// internal clock
	
	T1CON = 0x31;			// timer 1 ints
	TMR1IE = 1;
	PEIE = 1;

	T1CKPS0 = 1;			// prescale /8
	T1CKPS1 = 1;
	
	// setup timer0 (stepper step rate)
	T08BIT = 0;				// 16 bit
	PSA = 1;				// no prescaler
	T0PS0 = 1;				// preset prescaler to * 256 (we use this later)
	T0PS1 = 1;
	T0PS2 = 1;
	T0CS = 0;				// internal clock
	TMR0H = 0xff;
	TMR0L = 0xff;
	TMR0ON = 0;				// disable
	
	// global int enable
	GIE = 1;
	ei();
	
	LCDInit();
	DelayMs(250);
	LCDClear();
}


//------------------------------
// handle ints

int gHalfSecond = 0;

u16	gStepsNextInt = 0;
int	gNoPrescaler = 1;

// provide direct (half step) drive outputs
int	gDriveStepIndex = 0;
const u8	gHDriveSteps[8] = {0x50, 0x40, 0x60, 0x20, 0xa0, 0x80, 0x90, 0x10};

void interrupt IntHandler(void)
{
	//------------------------
	// timer 0 used to drive stepper
	if ( TMR0IE && TMR0IF )
	{
		// use the values we precalculated on the last int
		// need to get the timer going asap
		PSA = gNoPrescaler;
		TMR0H = gStepsNextInt >> 8;
		TMR0L = gStepsNextInt & 0xff;
		
		TMR0IF = 0;				// ack int 0

		// now calc values for next time.
		
		// decide if we need the prescaler or not...
		if (gStepRate < 160)
		{
			// prescaler needed
			gNoPrescaler = 0;
			gStepsNextInt = 0 - ((u16) (39063 / gStepRate));	// 39063 = 10mhz / prescaler
		
		}
		else
		{
			// no prescaler
			gNoPrescaler = 1;
			gStepsNextInt = 0 - ((u16) ((u32) 10000000 / gStepRate));	// 10mhz
		}

		// anything to do?
		if (gSteps)
		{
			// clock pin low
			StepOut = gTable.mStepPolarity ? 1 : 0;
			gSteps--;	// dec number of steps
		
			// turn off direct out (give some time for transistors to switch off).
			DIRECTOUT = DIRECTOUT & 0xf;
			
			// step H drive output
			if (gCurDirection)
				gDriveStepIndex--;
			else
				gDriveStepIndex--;
			gDriveStepIndex &= 7;

			// clock pin high... (0.5uS min)
			StepOut = gTable.mStepPolarity ? 0 : 1;

			// H drive direct output (top 4 bits)
			DIRECTOUT = gHDriveSteps[gDriveStepIndex] | (DIRECTOUT & 0xf);

			// ramp up or down?
			if (gAccelSteps)
			{
				gAccelSteps--;
				gStepRate += gStepRateRamp;
			}
			else
			{
				if (gSteps <= gDecelSteps)
				{
					gStepRate -= gStepRateRamp;
				}					
			}
		}
		// nothing to do - turn off timer, no more interupts until
		// we're re-enabled in StartMotor.
		else
		{
			TMR0ON = 0;			// turn off timer
			gComplete = 1;		// mark as complete (table at dest)
		}
	}

	//------------------------
	// handle timer1 int
	
	if ( TMR1IE && TMR1IF )
	{
		SET_TIMER(TIMER_RATE);

		TMR1IF = 0;
		gTimerCount++;
		gHalfSecond++;
		
		if (gHalfSecond == 10)	// 20hz rate
		{		
			gProcessTimer = true;
			gHalfSecond = 0;
		}
	}
}


//----------------------------------

void	Beep(int freq, int dur)
{
	int i;

	// only if ints aren't running
	if (gComplete)
	{
		for (i = 0; i < dur; i++)
		{
			SounderOut = 1;
			DelayUs(freq);		
			SounderOut = 0;
			DelayUs(freq);
		}
	}
}

//----------------------------------
// write a byte of data to EEPROM

void	EEWrite(u8 addr, u8 data)
{
	EEADR = addr;
	EEDATA = data;
	EEPGD = 0;			// access eeprom (1 = program)
	CFGS = 0;			// eeprom memory
	WREN=1;				// enable writes

	GIE=0;				// disable interrupts
	EECON2=0x55;		// required sequence for EEPROM update
	EECON2=0xAA;
	WR=1;
	
	while(WR)continue;
	
	EEIF=0;
	WREN=0;
	
	GIE=1;	// re-enable interrupts
}

//----------------------------------
// read a byte of data from EEPROM

u8	EERead(u8 addr)
{
	u8	eeprom_data;// in PIC18Fxx2 devices, the data needs
					// to be collected in the next instruction
					// following RD=1. A temporary variable is
					// created to do this.
	EEADR = addr;
	GIE = 0;
	EEPGD = 0;		// data memory
	CFGS = 0;		// data memory
	RD = 1;
	eeprom_data = EEDATA;
	GIE=1;
	return eeprom_data;
}


