#include "globals.h"
#include "display.h"
#include "lcd.h"
#include "keypad.h"
#include "main.h"
#include "delay.h"

#include "table.h"

// menu enum
enum
{
	SELECTFUNCTION,
	
	BEGIN_JOG,
	DO_JOG,
	
	BEGIN_DIVISION,
	DO_DIVISION,
	
	BEGIN_DEGREES,
	DO_DEGREES,

	BEGIN_CONTINUOUS,
	DO_CONTINUOUS,
	RUNNNING,

	BEGIN_SETUP,
	DO_SETUP
};


// setup enum
enum
{
	SET_FIRST,
	
	SET_MINSPEED = 0,
	SET_MAXSPEED,
	SET_WORMRATIO,
	SET_DIRECTION,
	SET_BACKLASH,
	SET_CLOCKPOLARITY,
	SET_PROFILE,
	SET_SAVEANDEXIT,
	SET_DEFAULTS,
	
	SET_LAST
};

//----------------------------------

int	gSubMode;
int	gSetup;

int	gCurDirection;
int	gCurrentSpeed;		// 0-4 = slow-fast

s32 gCurrentStepPosition;
s32	gStepsPerCycle;

s16	gCurrentDivision;
s32	gCurrentDegrees;

int	gCurProfile;

//----------------------------------
// global instance

TableData	gTable;

//----------------------------------

void	LoadData(int profile)
{
	char *ptr;
	int	size;
	int	i;

	// allow 32 bytes per profile
	int offset = profile << 5;

	size = sizeof(TableData);
	ptr = (char *) &gTable;

	for (i = 0; i < size; i++, ptr++)
		*ptr = EERead(i + offset);
}

//----------------------------------

void	SaveData(int profile)
{
	char *ptr;
	int	size;
	int	i;

	// allow 32 bytes per profile
	int offset = profile << 5;

	size = sizeof(TableData);
	ptr = (char *) &gTable;

	for (i = 0; i < size; i++, ptr++)
		EEWrite(i + offset, *ptr);
}

//----------------------------------

void	SetDefaults(void)
{
	gTable.mDataValid = 0xDEF0;

	gTable.mSlowMin = 200;
	gTable.mSlowMax = 500;
	gTable.mSlowRamp = 8;
	gTable.mFastMin = 350;
	gTable.mFastMax = 1000;
	gTable.mFastRamp = 8;
	gTable.mDirectionSwap = 1;
	gTable.mBacklash = 10;
	gTable.mDefaultSpeeds[0] = 0;
	gTable.mDefaultSpeeds[1] = 4;
	gTable.mDefaultSpeeds[2] = 4;
	gTable.mDefaultSpeeds[3] = 1;
	gTable.mNumDivisions = 60;
	gTable.mNumDegrees = 3000;
	gTable.mWormRatio = 90;
	gTable.mStepPolarity = 1;
}

//----------------------------------

void	InitTable()
{
	gCurProfile = 10;
	
	// prompt for profile
	LCDClear();
	LCDPuts("Enter Profile?");
	LCDGoto(2, 0);
	LCDPuts("(1 to 5)");

	// get an input of 1 - 5
	while (gCurProfile > 5 || gCurProfile < 1)
	{
		gCurProfile = EditNumber(2, 10, 1, 1, -1);
	}

	// shift it 0 - 4
	gCurProfile--;

	LoadData(gCurProfile);

	// is data valid?	
	if (gTable.mDataValid != 0xDEF0)
	{
		// no so create defaults
		SetDefaults();

		// and save in uninitialised area
		SaveData(gCurProfile);
	}

	gStepsPerCycle = (s32) gTable.mWormRatio * 400;
	
	gCurrentStepPosition = 0;
}


//----------------------------------


void	StartMotor(u32 numSteps, int direction)
{
	gCurDirection = direction;

	// set direction pin
	if (direction)
		DirectionOut = !gTable.mDirectionSwap;
	else
		DirectionOut = gTable.mDirectionSwap;

	gSteps = numSteps;

	// interpolate for speed
	gStepRate = (((gTable.mFastMin - gTable.mSlowMin) * gCurrentSpeed) / 4) + gTable.mSlowMin;
	gStepRateMax = (((gTable.mFastMax - gTable.mSlowMax) * gCurrentSpeed) / 4) + gTable.mSlowMax;
	gStepRateRamp = (((gTable.mFastRamp - gTable.mSlowRamp) * gCurrentSpeed) / 4) + gTable.mSlowRamp;
	
	// number of steps of acceleration/deceleration
	gDecelSteps = gAccelSteps = (gStepRateMax - gStepRate) / gStepRateRamp;

	// however there may not be enough time to do the full range...
	if ((gDecelSteps + gAccelSteps) > (gSteps >> 1))
		gDecelSteps = gAccelSteps = gSteps >> 1;

	// set to 1 by interupt routine when we arrive at our destination
	gComplete = 0;

	// set initial rate
	/*
	gStepsNextInt = StepRateTable[gStepRate];
	if (gStepRate > 100)
		gNoPrescaler = 1;
	else
		gNoPrescaler = 0;
	*/
	if (gStepRate < 160)
	{
		// prescaler needed
		gNoPrescaler = 0;
		gStepsNextInt = 0 - ((u16) (39063 / gStepRate));
	}
	else
	{
		// no prescaler
		gNoPrescaler = 1;
		gStepsNextInt = 0 - ((u16) ((u32) 10000000 / gStepRate));
	}

	// immediate interupt
	PSA = 1;
	TMR0H = 0xf0;
	TMR0L = 0xf0;

	
	// start timer
	TMR0IE = 1;
	TMR0ON = 1;
}


//----------------------------------

void	BeginTableMenus()
{
	LCDClear();
	//		  12345678901234567890
	LCDPuts("1> Jog      9> Setup\n");
	LCDPuts("2> Division\n");
	LCDPuts("3> Degrees\n");
	LCDPuts("4> Continuous");
		
	Mode = PROCESSTABLE;
	gSubMode = SELECTFUNCTION;
}

//----------------------------------

void	UseSpeed(int indx)
{
	if (gTable.mDefaultSpeeds[indx] > 4)
		gTable.mDefaultSpeeds[indx] = 4;

	gCurrentSpeed = gTable.mDefaultSpeeds[indx];

	LCDGoto(0, 14);
	LCDPuts("Spd: ");
	LCDWriteS16(gTable.mDefaultSpeeds[indx] + 1, 1, 0);
}

//----------------------------------

void	IncreaseSpeed(int indx)
{
	gTable.mDefaultSpeeds[indx]++;
	if (gTable.mDefaultSpeeds[indx] > 4)
		gTable.mDefaultSpeeds[indx] = 0;

	UseSpeed(indx);
}

//----------------------------------

void	DisplayPosition(s32 curpos, int line, int pos )
{
	u16	degrees;
	u16	fraction;
	
	LCDGoto(line, pos);
	
	degrees = (curpos * 360) / gStepsPerCycle;

	LCDWriteS16(degrees, 3, 1);
	LCDPutch('.');
	
	// num of steps in fractional part of degree
	fraction = curpos - ((degrees * gStepsPerCycle) / 360);
	
	fraction = (fraction * 360000) / gStepsPerCycle;
	
	LCDWriteS16(fraction, 3, 0);
}

//----------------------------------

void	DisplayCurrentPosition(void)
{
	LCDGoto(1, 0);
	LCDPuts("Pos: ");
	DisplayPosition(gCurrentStepPosition, 1, 5);
}

//----------------------------------
// val = 1/100ths of a degree

u32	DegreesToSteps(u32 degrees)
{
	return ((degrees * gStepsPerCycle) + 18000) / 36000;
}

//----------------------------------

int		WaitComplete(void)
{
	int ret = 0;	// 0 = OK, 1 = cancel, 2 = stop

	while (!gComplete)
	{
		// check for 'emergency' stop or cancel
		ScanKeypad();
		switch(DebouncedKeyASCII)
		{
			// emergency stop - kill ints and finish
			case	'X':
				TMR0ON = 0;
				gComplete = 1;
				gSteps = 0;
				ret = 2;
				break;

			// user cancelled - start deceleration process
			case	'C':
				TMR0ON = 0;
				gSteps = gDecelSteps;
				gAccelSteps = 0;
				TMR0ON = 1;
				ret = 1;
				break;
		}
		
	}
	return ret;
}

//----------------------------------

void	MoveSteps(u32 numSteps, int direction)
{
	// if CCW then add in overshoot for backlash
	if (!direction)
		numSteps += gTable.mBacklash;

	if (gSubMode != DO_CONTINUOUS)
		DisplayCurrentPosition();

	ActiveOut = 0;
	AckOut = 1;
	
	LCDGoto(3, 0);
	LCDPuts("Working...\n");

	StartMotor(numSteps, direction);

	// apply backlash unless emergency stop
	if (WaitComplete() != 2 && !direction)
		StartMotor(gTable.mBacklash, !direction);

	LCDGoto(3, 0);
	LCDPuts("          ");
	
	AckOut = 0;
	ActiveOut = 1;
}

//----------------------------------
// val = 1/100ths of a degree

void	AddDegreesToCurrentPosition(u16 val)
{
	u32	steps = DegreesToSteps(val);
	gCurrentStepPosition += steps;
	
	if ( gCurrentStepPosition >= gStepsPerCycle)
		gCurrentStepPosition -= gStepsPerCycle;
		
	MoveSteps(steps, 1);
}

//----------------------------------
// val = 1/100ths of a degree

void	SubDegreesFromCurrentPosition(u16 val)
{
	u32	steps = DegreesToSteps(val);
	gCurrentStepPosition -= steps;
	
	if ( gCurrentStepPosition < 0 )
		gCurrentStepPosition += gStepsPerCycle;
			
	MoveSteps(steps, 0);
}

//----------------------------------

void	AddStepsToCurrentPosition(u32 val)
{
	gCurrentStepPosition += val;
	
	if ( gCurrentStepPosition >= gStepsPerCycle )
		gCurrentStepPosition -= gStepsPerCycle;
		
	MoveSteps(val, 1);
}

//----------------------------------

void	SubStepsFromCurrentPosition(u32 val)
{
	gCurrentStepPosition -= val;
	
	if ( gCurrentStepPosition < 0 )
		gCurrentStepPosition += gStepsPerCycle;
	
	MoveSteps(val, 0);
}

//----------------------------------
// prompt for a direction to move in

int		GetDirection(int shortdir)
{
	LCDGoto(3, 0);
	//       12345678901234567890
	LCDPuts(" -Press  Direction-");

	while (1)
	{
		ScanKeypad();

		switch (DebouncedKeyASCII)
		{
			case	'N':
				gCurDirection = 1;
				return 1;
			case	'P':
				gCurDirection = 0;
				return 1;
			case	'O':
				gCurDirection = shortdir;
				return 1;
			case	'C':
				return 0;
		}

	}
	LCDClearLine(3,0,20);

	return 0;
}

//----------------------------------

void	DoJog(void)
{
	switch (DebouncedKeyASCII)
	{
		// +/- 100th deg.
		case	'3':	AddDegreesToCurrentPosition(1);			break;
		case	'1':	SubDegreesFromCurrentPosition(1);		break;

		// +/- 10th deg.
		case	'6':	AddDegreesToCurrentPosition(10);		break;
		case	'4':	SubDegreesFromCurrentPosition(10);		break;

		// +/- 1 deg.
		case	'9':	AddDegreesToCurrentPosition(100);		break;
		case	'7':	SubDegreesFromCurrentPosition(100);		break;
		
		// +/- 10 deg.
		case	'N':	AddDegreesToCurrentPosition(1000);		break;
		case	'P':	SubDegreesFromCurrentPosition(1000);	break;

		case	'S':
			IncreaseSpeed(0);
			break;

		case	'C':
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	PrintNumDivisions(void)
{
	LCDGoto(2, 0);
	LCDPuts("Div: ");
	LCDWriteS16(gCurrentDivision, 4, 1);
	LCDPuts(" of ");
	LCDWriteS16(gTable.mNumDivisions, 4, 1);
	LCDPuts("\n");
}

//----------------------------------

void	DrawDivisionMode(void)
{
	LCDClear();
	LCDPuts("Division:");
	PrintNumDivisions();
	DisplayCurrentPosition();
	UseSpeed(1);
}

//----------------------------------

void	GetNumDivisions(void)
{
	LCDClear();
	LCDPuts("Enter Number Divs:\n");
	LCDPuts("Divisions =     ");
	gTable.mNumDivisions = EditNumber(1, 12, 4, gTable.mNumDivisions, -1);
	gCurrentDivision = 0;
	gCurrentStepPosition = 0;
}

//----------------------------------

void	GotoDivision(s16 div, int dir)
{
	s32	steps;

	// fix any overflows
	if (div < 0)
		div += gTable.mNumDivisions;
	else
		if (div >= gTable.mNumDivisions)
			div -= gTable.mNumDivisions;

	gCurrentDivision = div;

	PrintNumDivisions();

	// absolute number of steps at this division
	steps = gStepsPerCycle * div / gTable.mNumDivisions;

	// number of steps from current position to get there
	// CW
	if (dir)
		steps = steps - gCurrentStepPosition;
	else
		steps = gCurrentStepPosition - steps;

	if (steps < 0)
		steps = steps + gStepsPerCycle;
	else
		if (steps >= gStepsPerCycle)
			steps = steps - gStepsPerCycle;

	// CW
	if (dir)
		AddStepsToCurrentPosition(steps);
	// CCW
	else
		SubStepsFromCurrentPosition(steps);
}

//----------------------------------

void	DoDivision(void)
{
	s16		div;
	s16		diff;
	int		dir;

	switch (DebouncedKeyASCII)
	{
		case	'P':
			GotoDivision(gCurrentDivision - 1, 0);
			break;
		case	'N':
			GotoDivision(gCurrentDivision + 1, 1);
			break;

		case	'O':
			div = EditNumber(2, 5, 4, gCurrentDivision, -1);
			if (!gEditCancelled)
			{
				// find a 'shortest route' direction
				diff = div + gTable.mNumDivisions - gCurrentDivision;
				if (diff > gTable.mNumDivisions)
					diff -= gTable.mNumDivisions;
	
				if (diff > (gTable.mNumDivisions >> 1))
					dir = 0;
				else
					dir = 1;
			
				if (div < gTable.mNumDivisions && GetDirection(dir))
					GotoDivision(div, gCurDirection);
			}
			break;

		case	'S':
			IncreaseSpeed(1);
			break;

		case	'C':
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	GetNumDegrees(void)
{
	LCDClear();
	LCDPuts("Enter Num Degrees:\n");
	LCDPuts("Degrees =      ");
	gTable.mNumDegrees = 6000; // default for now
	gTable.mNumDegrees = EditNumber(1, 10, 5, gTable.mNumDegrees, 3);

	while (gTable.mNumDegrees >= 36000)
		gTable.mNumDegrees -= 36000;

	gCurrentDegrees = 0;
	gCurrentStepPosition = 0;
}

//----------------------------------

void	PrintNumDegrees(void)
{
	u16	num;

	num = gTable.mNumDegrees / 100;

	LCDGoto(2, 0);
	LCDPuts("Deg: ");
	LCDWriteS16(num, 3, 1);
	LCDPuts(".");
	num = gTable.mNumDegrees - (num * 100);
	LCDWriteS16(num, 2, 0);
	LCDPuts("\n");
}

//----------------------------------

void	DrawDegreeMode(void)
{
	LCDClear();
	LCDPuts("Degree:");
	PrintNumDegrees();
	DisplayCurrentPosition();
	UseSpeed(2);
}

//----------------------------------

void	GotoDegrees(s32 deg, int dir)
{
	s32	steps;

	// fix any overflows
	if (deg < 0)
		deg += 36000;
	else
		if (deg >= 36000)
			deg -= 36000;

	gCurrentDegrees = deg;

	// absolute number of steps at this point
	steps = DegreesToSteps(gCurrentDegrees);

	// number of steps from current position to get there
	// CW
	if (dir)
		steps = steps - gCurrentStepPosition;
	else
		steps = gCurrentStepPosition - steps;

	if (steps < 0)
		steps = steps + gStepsPerCycle;
	else
		if (steps >= gStepsPerCycle)
			steps = steps - gStepsPerCycle;

	// CW
	if (dir)
		AddStepsToCurrentPosition(steps);
	// CCW
	else
		SubStepsFromCurrentPosition(steps);
}

//----------------------------------

void	DoDegrees(void)
{
	s32		deg;
	s32		diff;
	int		dir;

	switch (DebouncedKeyASCII)
	{
		case	'P':
			GotoDegrees(gCurrentDegrees - gTable.mNumDegrees, 0);
			break;
		case	'N':
			GotoDegrees(gCurrentDegrees + gTable.mNumDegrees, 1);
			break;

		case	'S':
			IncreaseSpeed(2);
			break;

		case	'O':
			deg = (s32) EditNumber(2, 5, 5, (u16) gTable.mNumDegrees, 3);
			if (!gEditCancelled)
			{
				// calc 'shortest route' direction
				diff = deg + (36000 - gCurrentDegrees);
				if (diff > 36000)
					diff -= 36000;
	
				if (diff > (36000 >> 1))
					dir = 0;
				else
					dir = 1;
	
				if (deg < 36000 && GetDirection(dir))
					GotoDegrees(deg, gCurDirection);
	
				DrawDegreeMode();
			}
			break;

		case	'C':
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	DoContinuous(void)
{
	switch (DebouncedKeyASCII)
	{
		case	'P':
			LCDClear();
			LCDPuts("Continuous:");
			UseSpeed(3);
			MoveSteps(0xfffff000, 0);		// 0xfffff000 - Warning - MoveSteps adds backlash!!
			gSubMode = BEGIN_CONTINUOUS;
			break;
		case	'N':
			LCDClear();
			LCDPuts("Continuous:");
			UseSpeed(3);
			MoveSteps(0xfffff000, 1);		// 0xfffff000 is a big number...
			gSubMode = BEGIN_CONTINUOUS;
			break;

		case	'S':
			IncreaseSpeed(3);
			break;

		case	'C':
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	DrawSetup(void)
{
	LCDClear();
	LCDGoto(0, 0);

	switch(gSetup)
	{
		case	SET_DEFAULTS:
			LCDPuts("Factory Defaults ?");
			LCDGoto(2, 0);
			LCDPuts("[G] resets.");
			break;

		case	SET_MINSPEED:
			LCDPuts("Set Min Speed ?");
			LCDGoto(2, 0);
			LCDPuts("Lo: ");
			LCDWriteS16(gTable.mSlowMin, 4, 1);
			LCDPuts(" Hi: ");
			LCDWriteS16(gTable.mSlowMax, 4, 1);
			LCDGoto(3, 0);
			LCDPuts("Ramp: ");
			LCDWriteS16(gTable.mSlowRamp, 2, 1);
			break;

		case	SET_MAXSPEED:
			LCDPuts("Set Max Speed ?");
			LCDGoto(2, 0);
			LCDPuts("Lo: ");
			LCDWriteS16(gTable.mFastMin, 4, 1);
			LCDPuts(" Hi: ");
			LCDWriteS16(gTable.mFastMax, 4, 1);
			LCDGoto(3, 0);
			LCDPuts("Ramp: ");
			LCDWriteS16(gTable.mFastRamp, 2, 1);
			break;

		case	SET_WORMRATIO:
			LCDPuts("Set Worm Ratio ?");
			LCDGoto(2, 0);
			LCDPuts("Ratio: ");
			LCDWriteS16(gTable.mWormRatio, 4, 1);
			break;

		case	SET_DIRECTION:
			LCDPuts("Swap Rotation Dir ?");
			LCDGoto(2, 0);
			if (gTable.mDirectionSwap)
				LCDPuts("Dir: CCW");
			else
				LCDPuts("Dir:  CW");
			break;

		case	SET_BACKLASH:
			LCDPuts("Set Backlash ?");
			LCDGoto(2, 0);
			LCDPuts("Num steps: ");
			LCDWriteS16(gTable.mBacklash, 3, 1);
			break;

		case	SET_CLOCKPOLARITY:
			LCDPuts("Swap Clock Pulse ?");
			LCDGoto(2, 0);
			if (gTable.mStepPolarity)
				LCDPuts("High  _-_\n");
			else
				LCDPuts("Low   -_-\n");
			break;

		case	SET_PROFILE:
			LCDPuts("Change Profile ?");
			LCDGoto(2, 0);
			LCDPuts("[G] to select new.");
			break;

		case	SET_SAVEANDEXIT:
			LCDPuts("Save and Exit ?");
			LCDGoto(2, 0);
			LCDPuts("[G] Saves.");
			break;
	}
}

//----------------------------------

void	ExecuteSetupOption(void)
{
	switch (gSetup)
	{
		case	SET_DEFAULTS:
			SetDefaults();
			BeginTableMenus();
			break;

		case	SET_MINSPEED:
			gTable.mSlowMin = EditNumber(2,4,4, gTable.mSlowMin, -1);
			gTable.mSlowMax = EditNumber(2,13,4, gTable.mSlowMax, -1);
			gTable.mSlowRamp = EditNumber(3,6,2, gTable.mSlowRamp, -1);
			if (gTable.mSlowMin < 1)
				gTable.mSlowMin = 1;
			if (gTable.mSlowMin > gTable.mSlowMax)
				gTable.mSlowMax = gTable.mSlowMin;
			if (gTable.mSlowRamp < 1)
				gTable.mSlowRamp = 1;
			break;
		case	SET_MAXSPEED:
			gTable.mFastMin = EditNumber(2,4,4, gTable.mFastMin, -1);
			gTable.mFastMax = EditNumber(2,13,4, gTable.mFastMax, -1);
			gTable.mFastRamp = EditNumber(3,6,2, gTable.mFastRamp, -1);
			if (gTable.mFastMin < 1)
				gTable.mFastMin = 1;
			if (gTable.mFastMin > gTable.mFastMax)
				gTable.mFastMax = gTable.mFastMin;
			if (gTable.mFastRamp < 1)
				gTable.mFastRamp = 1;
			break;
		case	SET_WORMRATIO:
			gTable.mWormRatio = EditNumber(2, 7, 4, gTable.mWormRatio, -1);
			gStepsPerCycle = (s32) gTable.mWormRatio * 400;
			gCurrentStepPosition = 0;
			break;
		case	SET_DIRECTION:
			gTable.mDirectionSwap = !gTable.mDirectionSwap;
			break;
		case	SET_BACKLASH:
			gTable.mBacklash = EditNumber(2, 10, 3, gTable.mBacklash, -1);
			break;
		case	SET_CLOCKPOLARITY:
			gTable.mStepPolarity = !gTable.mStepPolarity;
			break;
		case	SET_PROFILE:
			InitTable();
			break;
		case	SET_SAVEANDEXIT:
			SaveData(gCurProfile);
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	DoSetup(void)
{
	switch (DebouncedKeyASCII)
	{
		case 'O':
			ExecuteSetupOption();

			// if still in setup mode then redraw
			if (gSubMode == DO_SETUP)
				DrawSetup();
			break;

		case 'N':
			gSetup++;
			if (gSetup >= SET_LAST)
				gSetup = SET_FIRST;
			DrawSetup();
			break;

		case 'P':
			gSetup--;
			if (gSetup < SET_FIRST)
				gSetup = SET_LAST - 1;
			DrawSetup();
			break;

		case 'C':
			BeginTableMenus();
			break;
	}
}

//----------------------------------

void	ProcessTableMenus(void)
{
	switch(gSubMode)
	{
		case	SELECTFUNCTION:
			switch (DebouncedKeyASCII)
			{
				case	'1': gSubMode = BEGIN_JOG; break;
				case	'2': gSubMode = BEGIN_DIVISION; break;
				case	'3': gSubMode = BEGIN_DEGREES; break;
				case	'4': gSubMode = BEGIN_CONTINUOUS; break;
				case	'9': gSubMode = BEGIN_SETUP; break;
			}
			break;
			
		case	BEGIN_JOG:
			LCDClear();
			LCDPuts("Jog:");
			DisplayCurrentPosition();
			UseSpeed(0);
			gSubMode = DO_JOG;
			break;
			
		case	DO_JOG:
			DoJog();
			break;
			
		case	BEGIN_DIVISION:
			GetNumDivisions();
			if (gEditCancelled)
				BeginTableMenus();
			else
			{
				DrawDivisionMode();
				UseSpeed(1);
				gSubMode = DO_DIVISION;
			}
			break;
			
		case	DO_DIVISION:
			DoDivision();
			break;

		case	BEGIN_DEGREES:
			GetNumDegrees();
			if (gEditCancelled)
				BeginTableMenus();
			else
			{
				DrawDegreeMode();
				UseSpeed(2);
				gSubMode = DO_DEGREES;
			}
			break;

		case	DO_DEGREES:
			DoDegrees();
			break;

		case	BEGIN_CONTINUOUS:
			LCDClear();
			LCDPuts("Continuous:");
			LCDGoto(2,0);
			LCDPuts("Press direction");
			UseSpeed(3);
			gSubMode = DO_CONTINUOUS;
			break;

		case	DO_CONTINUOUS:
			DoContinuous();
			break;

		case	BEGIN_SETUP:
			gSetup = SET_FIRST;
			LCDClear();
			LCDPuts("Setup:");
			DrawSetup();
			gSubMode = DO_SETUP;
			break;

		case	DO_SETUP:
			DoSetup();
			break;
			
	}
}
