%clc;
clear all;

%%
% CNC Machine Servo motor Sizing by Jonathan Blissett

% References:
% http://www.nskamericas.com/cps/rde/xbcr/mx_es/Friction_Torque_and_Drive_Torque.pdf
% http://www.hiwin.com/pdf/bs/ballscrews.pdf
mu_rail = 0.2;%0.01;                        % Rail friction coefficient

% Machine spec [x y z]
m = [150 180 250];                      % Mass of carriage
F = [500 500 500];                      % Force, N
a = [2 2 1];                            % Required acceleration, m/s^2

L = [5 5 5];                            % Ballscrew pitch, mm
length = [0.935 0.45 0.6];              % Ballscrew length, m
d = [25 25 32];                         % Ballscrew diameter, mm

db = [2.381 2.381 3.175];               % Ballscrew ball diameter, mm
Ca = [699 629 780]*9.81;                % Ballscrew dynamic load rating, N
epsilon = 0.05;                         % Ballscrew pre-load force factor of Ca
                                        % 0.03-0.07 for cutting machine
                                        % tools, <=0.1 for especially high
                                        % rigidity

P = epsilon*Ca;                         % Ballscrew pre-load force                               

% MM9306WI2H are 750N/um 
fixity = [1.55 0.32 0.32];              % 2.24 for fixed-fixed 1.55 for fixed-simple, 1.0 for simple-simple, 0.32 for fixed-free
an.alpha = [60 60 40]*pi/180;           % Angular contact bearing support angle, suffix B is 40 degree, AC is 25 degree, C is 15 degree
an.Fpre = [1400 1400 735];                % Angular contact beraing preload force
an.Dw = [5/16*25.4 5/16*25.4 7];                        % Angular contact bearing ball diameter
an.Z = [12 12 11];                         % Number of balls

K = [62 62 74]*9.8;                     % Ballscrew (nut) stiffness factor, 35 for 1605, 56 for 2005, 62 for 2505 double

ballscrew_density = 7850;               % Ballscrew density, steel=7850kg/m^3
ba.mu = 0.006;                          % Ballscrew friction co-effficient Typical=0.003-0.1

% End of Spec

critical_speed = 0.8.*fixity*4.76*10^6.*(d-db)./25.4./((length.*1000./25.4).^2); 

ba.alpha = atan(L./(pi*d));             % Ballscrew lead angle
ba.beta = atan(ba.mu);                  % Ballscrew friction angle
n_1 = tan(ba.alpha)./tan(ba.alpha+ba.beta);% Ballscrew efficiency (common transmission)
n_2 = tan(ba.alpha-ba.beta)./tan(ba.alpha);% Ballscrew efficiency (reverse transmission)
k.p = 1./n_1-n_2;                       % Pre-load torque co-efficient


% Pulley specifications
inc_drive_pulley.inertia = [1 1 1];     % Set to 1 to include pulley inertia in calc, or 0 to exclude
inc_driven_pulley.inertia = [1 1 1];
pulley.pitch = 0.375*25.4               % Pulley pitch, mm
belt.width = 0.5*25.4;                      % Belt width, mm
pulley.drive_t = [20 12 18];            % No. of teeth on pulley coupled to motor shaft
pulley.drive_tn = [24 24 18];           % No. of teeth on driven pulley
pulley.density = 7850;                  % Density of pulley material, steel=7850kg/m^3
width_add = 10;                         % Difference between belt and pulley width, mm
pitch_dia_offset = 1;                   % Difference between pitch radius and pulley radius, mm
belt.span = [150 150 150];              % Belt span length (for stiffness calc)

% Motor specifications
rotor.length=61.5;                      % Rotor length, mm
rotor.OD=59;
rotor.ID=56;
endbell_length=6;                       % Motor endbell length, mm
rotor.density=7850;
endbell_density=2700;
magnet.density=7400;
magnet.thickness=2.5;
magnet.width=9;
magnet.length=45;
magnet.n=14;

magnet.coverage=magnet.n*magnet.width/(rotor.ID*pi);
J.fluxring=0.5*rotor.density*pi*((rotor.OD/2000)^4-(rotor.ID/2000)^4)*rotor.length/1000;                           %
J.endbell=0.5*endbell_density*pi*(rotor.OD/2000)^4*endbell_length/1000;                            %
J.magnets=magnet.coverage*0.5*magnet.density*pi*((rotor.ID/2000)^4-((rotor.ID-2*magnet.thickness)/2000)^4)*magnet.length/1000;

J.motor=J.fluxring+J.endbell+J.magnets;
J.motor=840/1000/100/100; %Used to exclude motor inertia

rpm=1000;                               % Rated speed, rpm

pulley.drive_r=pulley.pitch.*pulley.drive_t./(2.*pi)./1000-pitch_dia_offset/1000;
pulley.drive_rn=pulley.pitch.*pulley.drive_tn./(2.*pi)./1000-pitch_dia_offset/1000;


% Friction load

F_friction=m*mu_rail*9.81;              % F=mu*C for X and Y
F_friction(3)=F(3)*mu_rail;             % C for Z-axis is cutting force        

T.preload=k.p.*P.*L./1000./(2.*pi);     % Ballscrew pre-load torque

% Calculate inertias

J.linear=m.*(L./(1000.*2.*pi)).^2;
J.pulley.drive=0.5*pulley.density.*pi.*(pulley.drive_r).^4.*(belt.width+width_add)/1000.*inc_drive_pulley.inertia;
J.pulley.driven=0.5*pulley.density.*pi.*(pulley.drive_rn).^4.*(belt.width+width_add)/1000.*inc_driven_pulley.inertia;
J.ballscrew=0.5*ballscrew_density*pi.*(d/2000).^4.*length;

J.total=J.motor+J.pulley.drive+(pulley.drive_t./pulley.drive_tn).^2.*(J.linear+J.pulley.driven+J.ballscrew);

alpha=pulley.drive_tn/pulley.drive_t*2*pi*a/(L/1000);

% Calculate torques
F(3)=F(3)+m(3)*9.81; % Z force = cutting force + weight
T.load=T.preload+(F+F_friction).*L./1000./(2.*pi.*n_1);

T.total=pulley.drive_t./pulley.drive_tn.*T.load+J.total*alpha;

feedrate = rpm*L/1000.*pulley.drive_t./pulley.drive_tn;

fprintf('Feedrate in m/min:\t[x y z]=[%.1f %.1f %.1f]\nInertia in g-m^2:\t[x y z]=[%.2f %.2f %.2f]\nTorque in Nm:\t\t[x y z]=[%.2f %.2f %.2f]\nInertia ratios:\t\t[x y z]=[%.2f %.2f %.2f]\nCritical vel m/min\t[x y z]=[%.1f %.1f %.1f]',feedrate,1000*J.total,T.total,J.total/J.motor,critical_speed.*L/1000)

% Calculate ballscrew stiffness
% Reference: http://www.nskamericas.com/cps/rde/xbcr/na_en/Preload_and_Rigidity.pdf
% http://www.learneasy.info/MDME/MEMmods/class_projects/backstop/controller/Topic4-BallscrewCalculations.pdf


dr = (d-db)/1000;% Torsion
k.torsion = (pi*dr.^4/32*7.7e10)./length;
k.shaft = pi*dr.^2/4*2.06e5./length;       % Multiply stiffness by 4 for fixed-fixed
k.nut = 0.8*K.*(P./(0.1.*Ca)).^(1/3);
k.bearing = 3*an.Fpre./(0.44./sin(an.alpha).*((an.Fpre./an.Z.*sin(an.alpha)).^2./an.Dw).^(1/3));
k.total = 1./(1./k.shaft+1./k.nut+1./k.bearing);

fprintf('\n\nBallscrew stifnesses:\nShaft axial in N/um\t\t[x y z]=[%.1f %.1f %.1f]\nNut axial in N/um\t\t[x y z]=[%.1f %.1f %.1f]\nBearing axial in N/um\t[x y z]=[%.1f %.1f %.1f]\nOverall linear N/um\t\t[x y z]=[%.1f %.1f %.1f]',k.shaft,k.nut,k.bearing,k.total)

T.screw=T.total.*pulley.drive_t./pulley.drive_tn;
pulley.driven_prad=pulley.pitch.*pulley.drive_tn./(2.*pi)./1000;
belt.tension_force = T.screw./pulley.driven_prad;

belt.stiff = 23400./belt.span.*belt.width; %http://www.sgtransmisiones.com/info/pdf/correas/c_poliuretano.pdf
belt.elongation = belt.tension_force./belt.stiff; %http://www.bbman.com/assets/files/pdf-library/Engineering/Timing%20Belts/BeltTensileProperties.pdf
k.belt = T.screw./(belt.elongation./(pulley.pitch.*pulley.drive_t)*2*pi);
k.ttot = k.belt.*k.torsion./(k.belt+k.torsion);

fprintf('\n\nShaft torsion in Nm/rad\t[x y z]=[%.1f %.1f %.1f]',k.torsion)
fprintf('\nBelt stiffness Nm/rad\t[x y z]=[%.1f %.1f %.1f]',k.belt)
fprintf('\nTotal torsion in Nm/rad\t[x y z]=[%.1f %.1f %.1f]',k.ttot)

fprintf('\n\nAxial compliance error in um:\t\t[x y z]=[%.1f %.1f %.1f]',F./k.total)
fprintf('\nTorsional compliance error in um:\t[x y z]=[%.1f %.1f %.1f]',L./(2*pi)./k.ttot*1000.*T.screw)

%% Plot for stepper motor
% Look at http://homepage.cs.uiowa.edu/~jones/step/physics.html
pts = 100;

motor.holdtorque = 12;%3;%4;                     % Motor rated holding torque
motor.L = 9e-03;%3e-3;                     % Phase inductance
motor.Irated = 6.2;%4                             % Motor RMS current rating
motor.Isetpoint = 6.2;%4                          % Driver PEAK current setting
motor.detent = 0.02;
motor.voltage = 75;
motor.steps = 200;
motor.R = 0.75;%0.7;

motor.corner_w = 4/motor.steps*((motor.voltage^2-(motor.R*motor.Irated)^2)/((motor.L*motor.Irated)^2))^0.5;% V=jWLI+RI -> We=V/(LI) -> Wm=4V/(NLI) = Corner speed

%motor.corner_w2 = 4*motor.voltage/(motor.steps*motor.L*motor.Irated);% V=jWLI -> We=V/(LI) -> Wm=4V/(NLI) = Corner speed

motor.speed = [0 motor.corner_w motor.corner_w:2*motor.corner_w/(pts-3):3*motor.corner_w];
motor.torque = ones(1,pts);
motor.torque(1) = motor.holdtorque/sqrt(2)-motor.detent;     % Holding torque is less than rated as set to peak not rms current
motor.torque(2) = motor.torque(1);              % Same up to corner speed
motor.torque(3:end) = motor.torque(3:end)*motor.torque(2)./motor.speed(3:end)*motor.speed(2);

T.machine = ones(3,pts);
T.machine(1,:)=T.machine(1,:).*T.total(1);
T.machine(2,:)=T.machine(2,:).*T.total(2);
T.machine(3,:)=T.machine(3,:).*T.total(3);

%[index1 index2] = find(T.machine>[motor.torque' motor.torque' motor.torque']',1);

maxspeed = [motor.speed(find(T.machine(1,:)>motor.torque,1)) motor.speed(find(T.machine(2,:)>motor.torque,1)) motor.speed(find(T.machine(3,:)>motor.torque,1))];
%maxspeed = motor.speed(index);
maxfeed = maxspeed/pi*30.*L/1000.*pulley.drive_t./pulley.drive_tn;

disp([num2str(motor.voltage/(motor.L^0.5)) ' ' num2str(motor.corner_w*motor.torque(2))]) 

plot(motor.speed/pi*30,[motor.torque' T.machine'])
title(['Stepper motor torque vs Speed curve, Intersection at ' num2str(maxfeed) 'm/min'])
xlabel('Motor Speed, rpm')
ylabel('Torque, Nm')
xlim([0 1000])
ylim([0 motor.holdtorque])
legend('Motor','X','Y','Z')

